import "pe"
import "elf"

rule FASM {
    meta:
        category = "compiler"
        description = "detects fasm using DOS stub"
        author = "malcat"
        reliability = 70

    strings:
        $stub = { 4D5A80000100000004001000FFFF00004001000000000000400000000000000000000000000000000000000000000000000000000000000000000000800000000E1FBA0E00B409CD21B8014CCD21546869732070726F6772616D2063616E6E6F742062652072756E20696E20444F53206D6F64652E0D0A24000000000000000050450000 }

    condition:
        $stub at 0
}

///////////////////////// OTHER LINKERS

rule TurboLinker {
    meta:
        name        = "TurboLinker"
        category    = "compiler"
        description = "Linked with TurboLinker"
        author      = "malcat"
        created     = "2023-05-12"
        reliability = 80
        tlp         = "TLP:WHITE"
        sample      = "2a64b59654dbf1a12b93a4892e1312807d2be8b396a0adc73ecbd51af2689be2"
    strings:
        $dosstub = { 546869732070726F6772616D206D7573742062652072756E20756E6465722057696E????0D0A24??00 } private
        
    condition:
        $dosstub in (0..1000)
}


///////////////////////// OTHER C COMPILERS

rule BorlandCpp {
    meta:
        category = "compiler"
        description = "detects Borland C/Cpp compiler"
        author = "malcat"
        reliability = 40

    strings:
        $ = {EB1066623A432B2B484F4F4B}
        $ = "borlndmm" ascii fullword

    condition:
        pe.section_index(".tls") >= 0 and all of them and pe.characteristics & 0xC == 0xC
}

rule Golang {
    meta:
        category = "compiler"
        description = "detects golang compiler"
        author = "malcat"
        reliability = 60

    strings:
        $ = "runtime.traceLocker.GoCreate" ascii fullword
        $ = "runtime.gopanic" ascii fullword
        $ = "runtime.rt0_go" ascii fullword
        $ = "runtime.sysmontick" ascii fullword
        $ = { (FB | FA | F0 | F1) FF FF FF 00 00 ?? (04 | 08) }
        $ = "runtime.GOMAXPROCS" ascii fullword
        $ = "runtime.GOROOT" ascii fullword
        $ = "runtime/internal/atomic.Load" ascii fullword
        $ = "runtime/internal/atomic.(*Uint64).Add" ascii fullword

    condition:
         4 of them
}


rule Rust {
    meta:
        category = "compiler"
        description = "detects Rust compiler"
        author = "malcat"
        reliability = 70

    strings:
        $ = "0x00010203040506070809101112131415161718192021222324252627282930313233343536373839404142434445464748495051525354555657585960616263646566676869707172737475767778798081828384858687888990919293949596979899" ascii fullword
        $ = "panicking.rs" ascii fullword
        $ = "rust_panic" ascii fullword
        $ = "attempted to index str up to maximum usize" ascii
        $ = "a formatting trait implementation returned an error" ascii
		$ = "{recursion limit reached}" ascii
        $ = "RUST_BACKTRACE" ascii
        $ = "internal error: entered unreachable code" ascii
        $ = "__rust_begin_short_backtrace" ascii
        $ = "BorrowMutError" ascii
        $ = "Box<dyn Any>" ascii
        $ = "called `Option::unwrap()` on a `None` value" ascii

    condition:
         5 of them
}

private rule Nim_strings {
	meta:
		description = "Detects common strings left in Nim compiled executables."
		author = "Alexandre Côté Cyr"
		date = "2024-03-06"
		license = "BSD 2-Clause"
        version = "1"

	strings:
		$error0 = "@value out of range" ascii wide
		$error1 = "@division by zero" ascii wide
		$error2 = "@over- or underflow" ascii wide
		$error3 = "@index out of bounds" ascii wide
		$fatal0 = "fatal.nim" ascii wide
		$fatal1 = "sysFatal" ascii wide

	condition:
		all of ($fatal*) or 2 of ($error*)
}

rule Nim_PE {
     meta:
        description = "Detects Nim compiled PE files."
        author = "Alexandre Côté Cyr"
        date = "2024-03-06"
		license = "BSD 2-Clause"
        version = "1"
		category = "compiler"
		reliability = 90

    strings:
		$NimMain = "NimMain"
		$PreMainInner = "PreMainInner"

    condition:
		pe.is_pe and (Nim_strings or pe.exports(/NimMain/) or
		(pe.number_of_symbols != 0 and (
			// A Nim function name appears in the symbol table
			$NimMain in (pe.pointer_to_symbol_table..pe.pointer_to_symbol_table + pe.number_of_symbols * 0x18) or
			$PreMainInner in (pe.pointer_to_symbol_table..pe.pointer_to_symbol_table + pe.number_of_symbols * 0x18)
		)))
}

rule Nim_ELF {
     meta:
        description = "Detects Nim compiled ELF files."
        author = "Alexandre Côté Cyr"
        date = "2024-03-06"
		license = "BSD 2-Clause"
        version = "1"
		category = "compiler"
		reliability = 90

    condition:
		uint32(0) == 0x464c457f and (Nim_strings or
			(elf.symtab_entries != 0 and (
				// A Nim function name appears in the symbol table
				for any sym in elf.symtab: (
					(sym.type == elf.STT_FILE and sym.name endswith ".nim") or
					(sym.type == elf.STT_FUNC and (sym.name contains "NimMain" or sym.name == "PreMainInner"))
				)
			))
		)
}


rule NETAppHost {
   meta:
      category = "compiler"
      description = "detects native .NET apphost bootstrap loader"
      author = "malcat"
      reliability = 80
   strings:
      $ = "\\apphost.pdb" ascii
      $ = "&apphost_version=" wide
      $ = "You must install .NET to run this application." wide
      $ = "https://aka.ms/dotnet-core-applaunch?" wide
      $ = "The required library %s does not support single-file apps." wide
      $ = "COREHOST_TRACEFILE" wide

   condition:
      (uint16(0) == 0x5a4d
      or uint16(0) == 0x457f
      or uint32(0) == 0xFEEDFACF
      ) and 4 of them
}



rule MinGW {
    meta:
        category = "compiler"
        description = "detects mingw compiler"
        author = "malcat"
        reliability = 60

    strings:
        $ = "Mingw runtime failure" ascii
        $ = "Mingw-w64 runtime failure" ascii
        $ = "../../gcc/gcc/config/i386/w32-shared-ptr.c" ascii

    condition:
        pe.section_index(".bss") >= 0 and
        any of them and 
        (
            for all i in (0..pe.number_of_sections - 1): (
                pe.sections[i].characteristics & 0x700000 != 0
            ) or
            pe.characteristics & 0xC == 0xC
        )
}


rule PellesC {
    meta:
        category = "compiler"
        description = "detects pellesC compiler"
        author = "DIE (Jason Hood <jadoxa@yahoo.com.au>)"
        reliability = 60

    strings:
        $ep = { 5589E56AFF68????????68????????64FF35000000006489250000000083EC0C }

    condition:
        $ep at pe.entry_point and
        pe.linker_version.major == 2 and pe.linker_version.minor == 50
}


///////////////////////// MSVC LINKERS BY LINKER VERSION

rule MSVC_6_linker {
    meta:
        category = "compiler"
        description = "detects used visual studio version based on linker information"
        author = "malcat"
        reliability = 60

    condition:
        pe.linker_version.major == 6 and pe.linker_version.minor == 0
}


rule MSVC_2002_linker {
    meta:
        category = "compiler"
        description = "detects used visual studio version based on linker information"
        author = "malcat"
        reliability = 60

    condition:
        pe.linker_version.major == 7 and pe.linker_version.minor < 10
}

rule MSVC_2003_linker {
    meta:
        category = "compiler"
        description = "detects used visual studio version based on linker information"
        author = "malcat"
        reliability = 60

    condition:
        pe.linker_version.major == 7 and pe.linker_version.minor >= 10
}

rule MSVC_2005_linker {
    meta:
        category = "compiler"
        description = "detects used visual studio version based on linker information"
        author = "malcat"
        reliability = 60

    strings:
        $dosstub = "This program cannot" ascii private

    condition:
        pe.linker_version.major == 8 and pe.linker_version.minor == 0 and $dosstub in (0..1000)
}


rule MSVC_2008_linker {
    meta:
        category = "compiler"
        description = "detects used visual studio version based on linker information"
        author = "malcat"
        reliability = 60

    condition:
        pe.linker_version.major == 9
}

rule MSVC_2010_linker {
    meta:
        category = "compiler"
        description = "detects used visual studio version based on linker information"
        author = "malcat"
        reliability = 60

    condition:
        pe.linker_version.major == 10 and pe.linker_version.minor == 0
}

rule MSVC_2012_linker {
    meta:
        category = "compiler"
        description = "detects used visual studio version based on linker information"
        author = "malcat"
        reliability = 60

    condition:
        pe.linker_version.major == 11 and pe.linker_version.minor == 0
}

rule MSVC_2013_linker {
    meta:
        category = "compiler"
        description = "detects used visual studio version based on linker information"
        author = "malcat"
        reliability = 60

    condition:
        pe.linker_version.major == 12 and pe.linker_version.minor == 0
}

rule MSVC_2015_linker {
    meta:
        category = "compiler"
        description = "detects used visual studio version based on linker information"
        author = "malcat"
        reliability = 60

    condition:
        pe.linker_version.major == 14 and pe.linker_version.minor == 0
}

rule MSVC_2017_linker {
    meta:
        category = "compiler"
        description = "detects used visual studio version based on linker information"
        author = "malcat"
        reliability = 60

    condition:
        pe.linker_version.major == 14 and pe.linker_version.minor > 0 and pe.linker_version.minor <= 16
}

rule MSVC_2019_linker {
    meta:
        category = "compiler"
        description = "detects used visual studio version based on linker information"
        author = "malcat"
        reliability = 60

    condition:
        pe.linker_version.major == 14 and pe.linker_version.minor > 20 and pe.linker_version.minor < 30
}

rule MSVC_2022_linker {
    meta:
        category = "compiler"
        description = "detects used visual studio version based on linker information"
        author = "malcat"
        reliability = 60

    condition:
        pe.linker_version.major == 14 and pe.linker_version.minor >= 30 
}


///////////////////////// MSVC COMPILERS BY RICH ID


rule MSVC_6_rich {
    meta:
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        pe.rich_signature.version(0x2636) or
        pe.rich_signature.version(0x2306) or
        pe.rich_signature.version(0x2354) or
        pe.rich_signature.version(0x1fe8)
}

rule MSVC_2002_rich {
    meta:
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        pe.rich_signature.version(0x24fa) or
        pe.rich_signature.toolid(0x3d) and (pe.rich_signature.toolid(0x1c) or pe.rich_signature.toolid(0x1d))
}

rule MSVC_2003_rich {
    meta:
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        (
            pe.rich_signature.version(0x0c05) or 
            pe.rich_signature.version(0x0fc3) or 
            pe.rich_signature.version(0x0883) or 
            pe.rich_signature.version(0x178e)
        ) and 
        (pe.rich_signature.toolid(0x5f) or pe.rich_signature.toolid(0x60))
}

rule MSVC_2005_rich {
    meta:
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        pe.rich_signature.version(0xc627) and 
        (pe.rich_signature.toolid(0x6d) or pe.rich_signature.toolid(0x6e))
}

rule MSVC_2008_rich {
    meta:
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        (
            pe.rich_signature.version(0x521e) or 
            pe.rich_signature.version(0x7809)
        ) and
        pe.rich_signature.toolid(0x91) and (pe.rich_signature.toolid(0x83) or pe.rich_signature.toolid(0x84) or pe.rich_signature.toolid(0x89))
}

//////////////////////////////////////////// Automatically generated, credits go to https://github.com/dishather/richprint/tree/master

rule msvs2022_v17_14_5_pre_1_0_rich {
    meta:
        name = "MSVS2022 v17.14.5-pre.1.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x8989) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_13_6_rich {
    meta:
        name = "MSVS2022 v17.13.6"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x87fa) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_13_5_rich {
    meta:
        name = "MSVS2022 v17.13.5"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x87f9) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_12_4_rich {
    meta:
        name = "MSVS2022 v17.12.4"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x8684) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_12_3_rich {
    meta:
        name = "MSVS2022 v17.12.3"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x8683) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_12_1_rich {
    meta:
        name = "MSVS2022 v17.12.1"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x8681) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_11_5_rich {
    meta:
        name = "MSVS2022 v17.11.5"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x854b) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_11_4_rich {
    meta:
        name = "MSVS2022 v17.11.4"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x8548) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_10_5_rich {
    meta:
        name = "MSVS2022 v17.10.5"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x8415) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_10_4_rich {
    meta:
        name = "MSVS2022 v17.10.4"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x8414) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_10_3_rich {
    meta:
        name = "MSVS2022 v17.10.3"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x8413) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_9_7_rich {
    meta:
        name = "MSVS2022 v17.9.7"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x82f3) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_9_3_rich {
    meta:
        name = "MSVS2022 v17.9.3"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x82f2) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_9_1_rich {
    meta:
        name = "MSVS2022 v17.9.1"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x82f0) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_8_6_rich {
    meta:
        name = "MSVS2022 v17.8.6"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x816f) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_8_5_rich {
    meta:
        name = "MSVS2022 v17.8.5"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x816e) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_8_3_rich {
    meta:
        name = "MSVS2022 v17.8.3"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x816d) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_8_2_rich {
    meta:
        name = "MSVS2022 v17.8.2"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x816a) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_7_6_rich {
    meta:
        name = "MSVS2022 v17.7.6"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x8039) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_7_4_rich {
    meta:
        name = "MSVS2022 v17.7.4"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x8038) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_7_3_rich {
    meta:
        name = "MSVS2022 v17.7.3"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x8036) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_6_5_rich {
    meta:
        name = "MSVS2022 v17.6.5"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7f19) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_6_4_rich {
    meta:
        name = "MSVS2022 v17.6.4"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7f17) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_6_3_rich {
    meta:
        name = "MSVS2022 v17.6.3"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7f16) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_6_2_rich {
    meta:
        name = "MSVS2022 v17.6.2"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7f14) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_5_5_rich {
    meta:
        name = "MSVS2022 v17.5.5"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7dd9) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_5_3_rich {
    meta:
        name = "MSVS2022 v17.5.3"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7dd8) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_5_2_rich {
    meta:
        name = "MSVS2022 v17.5.2"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7dd7) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_4_5_rich {
    meta:
        name = "MSVS2022 v17.4.5"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7cc6) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_4_3_rich {
    meta:
        name = "MSVS2022 v17.4.3"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7cc1) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_4_2_rich {
    meta:
        name = "MSVS2022 v17.4.2"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7cbf) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_3_4_rich {
    meta:
        name = "MSVS2022 v17.3.4"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7b8e) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_2_5_rich {
    meta:
        name = "MSVS2022 v17.2.5"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7a64) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_2_1___17_2_4_rich {
    meta:
        name = "MSVS2022 v17.2.1 - 17.2.4"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7a61) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_14_6_pre_1_0_rich {
    meta:
        name = "MSVS2022 v17.14.6-pre.1.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x898b) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_14_3_rich {
    meta:
        name = "MSVS2022 v17.14.3"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x8988) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_14_2_pre_1_0_rich {
    meta:
        name = "MSVS2022 v17.14.2-pre.1.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x8987) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_14_0_pre_6_0_rich {
    meta:
        name = "MSVS2022 v17.14.0-pre.6.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x8938) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_14_0_pre_5_0_rich {
    meta:
        name = "MSVS2022 v17.14.0-pre.5.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x8928) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_14_0_pre_3_0_rich {
    meta:
        name = "MSVS2022 v17.14.0-pre.3.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x8925) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_14_0_pre_2_0_rich {
    meta:
        name = "MSVS2022 v17.14.0-pre.2.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x8866) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_14_0_pre_1_1_rich {
    meta:
        name = "MSVS2022 v17.14.0-pre.1.1"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x8807) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_13_2_rich {
    meta:
        name = "MSVS2022 v17.13.2"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x87f8) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_13_0_pre_3_0_rich {
    meta:
        name = "MSVS2022 v17.13.0-pre.3.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x873a) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_13_0_pre_1_0_rich {
    meta:
        name = "MSVS2022 v17.13.0-pre.1.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x872c) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_12_0_pre_5_0_rich {
    meta:
        name = "MSVS2022 v17.12.0-pre.5.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x8680) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_12_0_pre_4_0_rich {
    meta:
        name = "MSVS2022 v17.12.0-pre.4.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x867f) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_12_0_pre_3_0_rich {
    meta:
        name = "MSVS2022 v17.12.0-pre.3.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x867e) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_12_0_pre_2_1_rich {
    meta:
        name = "MSVS2022 v17.12.0-pre.2.1"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x8611) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_12_0_pre_1_0_rich {
    meta:
        name = "MSVS2022 v17.12.0-pre.1.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x85b2) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_11_0_pre_7_0_rich {
    meta:
        name = "MSVS2022 v17.11.0-pre.7.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x8547) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_11_0_pre_6_0_rich {
    meta:
        name = "MSVS2022 v17.11.0-pre.6.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x8545) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_11_0_pre_4_0_rich {
    meta:
        name = "MSVS2022 v17.11.0-pre.4.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x84e5) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_11_0_pre_2_1_rich {
    meta:
        name = "MSVS2022 v17.11.0-pre.2.1"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x8483) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_11_0_pre_1_1_rich {
    meta:
        name = "MSVS2022 v17.11.0-pre.1.1"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x846d) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_10_0_rich {
    meta:
        name = "MSVS2022 v17.10.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x8410) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_10_0_pre_4_0_rich {
    meta:
        name = "MSVS2022 v17.10.0-pre.4.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x840f) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_10_0_pre_3_0_rich {
    meta:
        name = "MSVS2022 v17.10.0-pre.3.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x83b9) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_10_0_pre_2_0_rich {
    meta:
        name = "MSVS2022 v17.10.0-pre.2.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x8351) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_9_2_rich {
    meta:
        name = "MSVS2022 v17.9.2"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x82f1) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_9_0_rich {
    meta:
        name = "MSVS2022 v17.9.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x82ef) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_9_0_pre_4_0_rich {
    meta:
        name = "MSVS2022 v17.9.0-pre.4.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x8294) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_9_0_pre_2_1_rich {
    meta:
        name = "MSVS2022 v17.9.0-pre.2.1"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x8229) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_9_0_pre_1_1_rich {
    meta:
        name = "MSVS2022 v17.9.0-pre.1.1"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x81c2) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_8_0_pre_6_0_rich {
    meta:
        name = "MSVS2022 v17.8.0-pre.6.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x8169) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_8_0_pre_5_0_rich {
    meta:
        name = "MSVS2022 v17.8.0-pre.5.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x8168) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_8_0_pre_4_0_rich {
    meta:
        name = "MSVS2022 v17.8.0-pre.4.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x8166) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_8_0_pre_2_0_rich {
    meta:
        name = "MSVS2022 v17.8.0-pre.2.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x8106) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_8_0_pre_1_0_rich {
    meta:
        name = "MSVS2022 v17.8.0-pre.1.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x8097) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_7_0_pre_6_0_rich {
    meta:
        name = "MSVS2022 v17.7.0-pre.6.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x8034) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_7_0_pre_2_0_rich {
    meta:
        name = "MSVS2022 v17.7.0-pre.2.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7fc1) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_6_0_pre_7_0_rich {
    meta:
        name = "MSVS2022 v17.6.0-pre.7.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7f12) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_6_0_pre_4_0_rich {
    meta:
        name = "MSVS2022 v17.6.0-pre.4.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7f0a) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_6_0_pre_2_0_rich {
    meta:
        name = "MSVS2022 v17.6.0-pre.2.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7ef6) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_6_0_pre_1_0_rich {
    meta:
        name = "MSVS2022 v17.6.0-pre.1.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7e43) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_5_0_pre_4_0_rich {
    meta:
        name = "MSVS2022 v17.5.0-pre.4.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7dd5) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_5_0_pre_2_0_rich {
    meta:
        name = "MSVS2022 v17.5.0-pre.2.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7d7c) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_5_0_pre_1_0_rich {
    meta:
        name = "MSVS2022 v17.5.0-pre.1.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7d13) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_4_0_pre_6_0_rich {
    meta:
        name = "MSVS2022 v17.4.0-pre.6.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7cbd) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_4_0_pre_5_0_rich {
    meta:
        name = "MSVS2022 v17.4.0-pre.5.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7cbc) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_4_0_pre_4_0_rich {
    meta:
        name = "MSVS2022 v17.4.0-pre.4.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7cbb) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_4_0_pre_3_0_rich {
    meta:
        name = "MSVS2022 v17.4.0-pre.3.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7cb1) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_4_0_pre_2_0_rich {
    meta:
        name = "MSVS2022 v17.4.0-pre.2.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7c4f) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_4_0_pre_1_0_rich {
    meta:
        name = "MSVS2022 v17.4.0-pre.1.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7be9) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_3_0_rich {
    meta:
        name = "MSVS2022 v17.3.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7b8d) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_3_0_pre_4_0_rich {
    meta:
        name = "MSVS2022 v17.3.0-pre.4.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7b8c) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_3_0_pre_3_0_rich {
    meta:
        name = "MSVS2022 v17.3.0-pre.3.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7b8b) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_3_0_pre_2_0_rich {
    meta:
        name = "MSVS2022 v17.3.0-pre.2.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7b1d) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_3_0_pre_1_0_rich {
    meta:
        name = "MSVS2022 v17.3.0-pre.1.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7ac0) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_2_0_rich {
    meta:
        name = "MSVS2022 v17.2.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7a60) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_2_0_pre_3_0_rich {
    meta:
        name = "MSVS2022 v17.2.0-pre.3.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7a5e) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_2_0_pre_2_1_rich {
    meta:
        name = "MSVS2022 v17.2.0-pre.2.1"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7a46) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_2_0_pre_1_0_rich {
    meta:
        name = "MSVS2022 v17.2.0-pre.1.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x798a) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_1_0_pre_5_0_rich {
    meta:
        name = "MSVS2022 v17.1.0-pre.5.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7980) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_1_0_pre_3_0_rich {
    meta:
        name = "MSVS2022 v17.1.0-pre.3.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x797f) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_1_0_pre_2_0_rich {
    meta:
        name = "MSVS2022 v17.1.0-pre.2.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x78c7) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_1_0_pre_1_0_rich {
    meta:
        name = "MSVS2022 v17.1.0-pre.1.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7862) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_0_0_pre_7_0_rich {
    meta:
        name = "MSVS2022 v17.0.0-pre.7.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x77f1) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_0_0_pre_5_0_rich {
    meta:
        name = "MSVS2022 v17.0.0-pre.5.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x77f0) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_0_0_pre_4_0_rich {
    meta:
        name = "MSVS2022 v17.0.0-pre.4.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7740) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_0_0_pre_3_1_rich {
    meta:
        name = "MSVS2022 v17.0.0-pre-3.1"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x76d7) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2022_v17_0_0_preview2_rich {
    meta:
        name = "MSVS2022 v17.0.0-preview2"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x76c1) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2019_v16_11_48_rich {
    meta:
        name = "MSVS2019 v16.11.48"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x75cf) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2019_v16_11_44_rich {
    meta:
        name = "MSVS2019 v16.11.44"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x75ce) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2019_v16_11_42_rich {
    meta:
        name = "MSVS2019 v16.11.42"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x75cd) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2019_v16_11_41_rich {
    meta:
        name = "MSVS2019 v16.11.41"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x75cc) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2019_v16_11_40_rich {
    meta:
        name = "MSVS2019 v16.11.40"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x75ca) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2019_v16_11_33_rich {
    meta:
        name = "MSVS2019 v16.11.33"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x75c9) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2019_v16_11_31_rich {
    meta:
        name = "MSVS2019 v16.11.31"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x75c8) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2019_v16_11_29_rich {
    meta:
        name = "MSVS2019 v16.11.29"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x75c7) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2019_v16_11_26_rich {
    meta:
        name = "MSVS2019 v16.11.26"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x75c4) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2019_v16_11_21_rich {
    meta:
        name = "MSVS2019 v16.11.21"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x75c3) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2019_v16_11_17_rich {
    meta:
        name = "MSVS2019 v16.11.17"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x75c2) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2019_v16_11_15_rich {
    meta:
        name = "MSVS2019 v16.11.15"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x75c1) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2019_v16_11_14_rich {
    meta:
        name = "MSVS2019 v16.11.14"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x75c0) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2019_v16_11_13_rich {
    meta:
        name = "MSVS2019 v16.11.13"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x75bf) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2019_v16_11_12_rich {
    meta:
        name = "MSVS2019 v16.11.12"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x75be) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2019_v16_11_11_rich {
    meta:
        name = "MSVS2019 v16.11.11"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x75bd) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2019_v16_11_10_rich {
    meta:
        name = "MSVS2019 v16.11.10"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x75bc) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2019_v16_11_9_rich {
    meta:
        name = "MSVS2019 v16.11.9"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x75bb) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2019_v16_11_8_rich {
    meta:
        name = "MSVS2019 v16.11.8"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x75ba) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2019_v16_11_6_rich {
    meta:
        name = "MSVS2019 v16.11.6"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x75b9) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2019_v16_11_5_rich {
    meta:
        name = "MSVS2019 v16.11.5"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x75b8) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2019_v16_11_1_rich {
    meta:
        name = "MSVS2019 v16.11.1"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x75b5) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2019_v16_10_4_rich {
    meta:
        name = "MSVS2019 v16.10.4"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7558) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2019_v16_10_3_rich {
    meta:
        name = "MSVS2019 v16.10.3"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7556) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2019_v16_10_0_rich {
    meta:
        name = "MSVS2019 v16.10.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7555) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2019_v16_9_5_rich {
    meta:
        name = "MSVS2019 v16.9.5"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x74db) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2019_v16_9_4_rich {
    meta:
        name = "MSVS2019 v16.9.4"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x74da) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2019_v16_9_2_rich {
    meta:
        name = "MSVS2019 v16.9.2"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x74d9) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2019_v16_9_0_rich {
    meta:
        name = "MSVS2019 v16.9.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x74d6) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2019_v16_8_5_rich {
    meta:
        name = "MSVS2019 v16.8.5"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7299) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2019_v16_8_4_rich {
    meta:
        name = "MSVS2019 v16.8.4"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7298) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2019_v16_8_3_rich {
    meta:
        name = "MSVS2019 v16.8.3"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7297) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2019_v16_8_2_rich {
    meta:
        name = "MSVS2019 v16.8.2"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7296) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2019_v16_8_0_rich {
    meta:
        name = "MSVS2019 v16.8.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7295) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2019_v16_7_5_rich {
    meta:
        name = "MSVS2019 v16.7.5"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x71b8) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2019_v16_7_1____16_7_4_rich {
    meta:
        name = "MSVS2019 v16.7.1 .. 16.7.4"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x71b7) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2019_v16_7_0_rich {
    meta:
        name = "MSVS2019 v16.7.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x71b6) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2019_v16_6_2_____16_6_5_rich {
    meta:
        name = "MSVS2019 v16.6.2 ... 16.6.5"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7086) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2019_v16_6_0_rich {
    meta:
        name = "MSVS2019 v16.6.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7085) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2019_v16_5_5_rich {
    meta:
        name = "MSVS2019 v16.5.5"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x6fc6) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule visual_studio_2019_version_16_5_2_rich {
    meta:
        name = "Visual Studio 2019 version 16.5.2"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x6fc4) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule visual_studio_2019_version_16_5_1_rich {
    meta:
        name = "Visual Studio 2019 version 16.5.1"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x6fc3) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule visual_studio_2019_version_16_5_0_rich {
    meta:
        name = "Visual Studio 2019 version 16.5.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x6fc2) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2019_v16_4_6_rich {
    meta:
        name = "MSVS2019 v16.4.6"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x6e9f) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2019_v16_4_4_rich {
    meta:
        name = "MSVS2019 v16.4.4"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x6e9c) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2019_v16_4_3_rich {
    meta:
        name = "MSVS2019 v16.4.3"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x6e9b) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule visual_studio_2019_version_16_4_0_rich {
    meta:
        name = "Visual Studio 2019 version 16.4.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x6e9a) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule visual_studio_2019_version_16_3_2_rich {
    meta:
        name = "Visual Studio 2019 version 16.3.2"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x6dc9) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule visual_studio_2019_version_16_2_3_rich {
    meta:
        name = "Visual Studio 2019 version 16.2.3"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x6d01) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule visual_studio_2019_version_16_1_2_rich {
    meta:
        name = "Visual Studio 2019 version 16.1.2"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x6c36) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs2019_v16_0_0_rich {
    meta:
        name = "MSVS2019 v16.0.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x6b74) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule visual_studio_2017_version_15_9_11_rich {
    meta:
        name = "Visual Studio 2017 version 15.9.11"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x6996) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule visual_studio_2017_version_15_9_7_rich {
    meta:
        name = "Visual Studio 2017 version 15.9.7"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x6993) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule visual_studio_2017_version_15_9_5_rich {
    meta:
        name = "Visual Studio 2017 version 15.9.5"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x6992) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule visual_studio_2017_version_15_9_4_rich {
    meta:
        name = "Visual Studio 2017 version 15.9.4"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x6991) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule visual_studio_2017_version_15_9_1_rich {
    meta:
        name = "Visual Studio 2017 version 15.9.1"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x698f) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule visual_studio_2017_version_15_8_5_rich {
    meta:
        name = "Visual Studio 2017 version 15.8.5"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x686c) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule visual_studio_2017_version_15_8_9_rich {
    meta:
        name = "Visual Studio 2017 version 15.8.9"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x686a) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule visual_studio_2017_version_15_8_4_rich {
    meta:
        name = "Visual Studio 2017 version 15.8.4"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x6869) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule visual_studio_2017_version_15_8_0_rich {
    meta:
        name = "Visual Studio 2017 version 15.8.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x6866) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule visual_studio_2017_version_15_7_5_rich {
    meta:
        name = "Visual Studio 2017 version 15.7.5"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x6741) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule visual_studio_2017_version_15_7_4_rich {
    meta:
        name = "Visual Studio 2017 version 15.7.4"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x673f) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule visual_studio_2017_version_15_7_3_rich {
    meta:
        name = "Visual Studio 2017 version 15.7.3"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x673e) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule visual_studio_2017_version_15_7_2_rich {
    meta:
        name = "Visual Studio 2017 version 15.7.2"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x673d) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule visual_studio_2017_version_15_7_1_rich {
    meta:
        name = "Visual Studio 2017 version 15.7.1"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x673c) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule visual_studio_2017_version_15_6_7_rich {
    meta:
        name = "Visual Studio 2017 version 15.6.7"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x6614) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule visual_studio_2017_version_15_6_6_rich {
    meta:
        name = "Visual Studio 2017 version 15.6.6"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x6613) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule visual_studio_2017_version_15_6_3_rich {
    meta:
        name = "Visual Studio 2017 version 15.6.3"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x6611) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule visual_studio_2017_version_15_6_0_rich {
    meta:
        name = "Visual Studio 2017 version 15.6.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x6610) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule visual_studio_2017_version_15_5_6_rich {
    meta:
        name = "Visual Studio 2017 version 15.5.6"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x64eb) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule visual_studio_2017_version_15_5_2_rich {
    meta:
        name = "Visual Studio 2017 version 15.5.2"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x64e7) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule visual_studio_2017_version_15_4_5_rich {
    meta:
        name = "Visual Studio 2017 version 15.4.5"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x63cb) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule visual_studio_2017_version_15_4_4_rich {
    meta:
        name = "Visual Studio 2017 version 15.4.4"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x63c6) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule visual_studio_2017_version_15_3_3_rich {
    meta:
        name = "Visual Studio 2017 version 15.3.3"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x63a3) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule visual_studio_2017_version_15_3_rich {
    meta:
        name = "Visual Studio 2017 version 15.3"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x63a2) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule visual_studio_2017_version_15_0_rich {
    meta:
        name = "Visual Studio 2017 version 15.0"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x61b9) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs_2015_upd3_1_rich {
    meta:
        name = "MSVS 2015 UPD3.1"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x5e97) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs_2015_upd3_rich {
    meta:
        name = "MSVS 2015 UPD3"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x5e95) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule visual_studio_2015_update_3__14_0__rich {
    meta:
        name = "Visual Studio 2015 Update 3 [14.0]"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x5e92) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs_2015_upd2_rich {
    meta:
        name = "MSVS 2015 UPD2"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x5d6e) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs_2015_14_0_24728_2_rich {
    meta:
        name = "MSVS 2015 14.0.24728.2"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x5bd2) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule msvs_2015__14_0__rich {
    meta:
        name = "MSVS 2015 [14.0]"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x59f2) ) and
        pe.rich_signature.toolid(0x102) and (pe.rich_signature.toolid(0x104) or pe.rich_signature.toolid(0x105))
}


rule visual_studio_2013_november_ctp__12_0__rich {
    meta:
        name = "Visual Studio 2013 November CTP [12.0]"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x527a) ) and
        pe.rich_signature.toolid(0xde) and (pe.rich_signature.toolid(0xe0) or pe.rich_signature.toolid(0xe1))
}


rule msvs2013_12_0_40629_00_update_5_rich {
    meta:
        name = "MSVS2013 12.0.40629.00 Update 5"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x9eb5) ) and
        pe.rich_signature.toolid(0xde) and (pe.rich_signature.toolid(0xe0) or pe.rich_signature.toolid(0xe1))
}


rule visual_studio_2013_update2_rc__12_0__rich {
    meta:
        name = "Visual Studio 2013 Update2 RC [12.0]"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x7674) ) and
        pe.rich_signature.toolid(0xde) and (pe.rich_signature.toolid(0xe0) or pe.rich_signature.toolid(0xe1))
}


rule visual_studio_2013_update_1__12_0__also_has_this_build_number_rich {
    meta:
        name = "Visual Studio 2013 Update 1 [12.0] also has this build number"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x520d) ) and
        pe.rich_signature.toolid(0xde) and (pe.rich_signature.toolid(0xe0) or pe.rich_signature.toolid(0xe1))
}


rule visual_studio_2013_rc__12_0__rich {
    meta:
        name = "Visual Studio 2013 RC [12.0]"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x515b) ) and
        pe.rich_signature.toolid(0xde) and (pe.rich_signature.toolid(0xe0) or pe.rich_signature.toolid(0xe1))
}


rule visual_studio_2013_preview__12_0__rich {
    meta:
        name = "Visual Studio 2013 Preview [12.0]"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x5089) ) and
        pe.rich_signature.toolid(0xde) and (pe.rich_signature.toolid(0xe0) or pe.rich_signature.toolid(0xe1))
}


rule msvs2012_premium_update_4_rich {
    meta:
        name = "MSVS2012 Premium Update 4"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0xee66) ) and
        pe.rich_signature.toolid(0xcc) and (pe.rich_signature.toolid(0xce) or pe.rich_signature.toolid(0xcf))
}


rule visual_studio_2012_november_ctp__11_0__rich {
    meta:
        name = "Visual Studio 2012 November CTP [11.0]"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0xc751) ) and
        pe.rich_signature.toolid(0xcc) and (pe.rich_signature.toolid(0xce) or pe.rich_signature.toolid(0xcf))
}


rule msvs2012_premium_rich {
    meta:
        name = "MSVS2012 Premium"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0xc627) ) and
        pe.rich_signature.toolid(0xcc) and (pe.rich_signature.toolid(0xce) or pe.rich_signature.toolid(0xcf))
}


rule msvs2010_sp1_kb_983509_rich {
    meta:
        name = "MSVS2010 SP1 kb 983509"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x9d1b) ) and
        pe.rich_signature.toolid(0x9d) and (pe.rich_signature.toolid(0xaa) or pe.rich_signature.toolid(0xab))
}


rule msvs2010_rich {
    meta:
        name = "MSVS2010"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x766f) ) and
        pe.rich_signature.toolid(0x9d) and (pe.rich_signature.toolid(0xaa) or pe.rich_signature.toolid(0xab))
}


rule visual_studio_2010_beta_2__10_0__rich {
    meta:
        name = "Visual Studio 2010 Beta 2 [10.0]"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x520b) ) and
        pe.rich_signature.toolid(0x9d) and (pe.rich_signature.toolid(0xaa) or pe.rich_signature.toolid(0xab))
}


rule visual_studio_2010_beta_1__10_0__rich {
    meta:
        name = "Visual Studio 2010 Beta 1 [10.0]"
        category = "compiler"
        description = "detects used visual studio version based on rich header information"
        author = "malcat"
        reliability = 80

    condition:
        ( pe.rich_signature.version(0x501a) ) and
        pe.rich_signature.toolid(0x9d) and (pe.rich_signature.toolid(0xaa) or pe.rich_signature.toolid(0xab))
}

